<?php

require_once '../core/config.php';

try {
    // Connect to MySQL server without specifying a database
    $pdo = new PDO("mysql:host=" . DB_HOST, DB_USER, DB_PASS);
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);

    // Create the database if it doesn't exist
    $pdo->exec("CREATE DATABASE IF NOT EXISTS `" . DB_NAME . "` CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci");
    echo "Database '" . DB_NAME . "' created or already exists.<br>";

    // Now connect to the specific database
    $pdo = new PDO("mysql:host=" . DB_HOST . ";dbname=" . DB_NAME, DB_USER, DB_PASS);
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);

    echo "Connected to database '" . DB_NAME . "' successfully.<br>";

    // SQL statements to create tables
    $sql = "
        -- Users Table: Stores all individuals in the system
        CREATE TABLE IF NOT EXISTS users (
            id INT AUTO_INCREMENT PRIMARY KEY,
            email VARCHAR(255) NOT NULL UNIQUE,
            username VARCHAR(50) NOT NULL UNIQUE,
            password_hash VARCHAR(255) NOT NULL,
            first_name VARCHAR(100),
            last_name VARCHAR(100),
            profile_picture VARCHAR(255) DEFAULT 'default.png',
            bio TEXT,
            role ENUM(
                'fan', 'player', 'team_manager', 'club_manager', 
                'league_manager', 'referee', 'commissioner', 
                'admin', 'super_admin'
            ) NOT NULL DEFAULT 'fan',
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
        ) ENGINE=InnoDB;

        -- Seasons Table
        CREATE TABLE IF NOT EXISTS seasons (
            id INT AUTO_INCREMENT PRIMARY KEY,
            name VARCHAR(50) NOT NULL UNIQUE, -- e.g., '2023/2024'
            is_active BOOLEAN DEFAULT TRUE,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
        ) ENGINE=InnoDB;

        -- Leagues Table
        CREATE TABLE IF NOT EXISTS leagues (
            id INT AUTO_INCREMENT PRIMARY KEY,
            name VARCHAR(255) NOT NULL,
            country VARCHAR(100),
            season VARCHAR(50) NOT NULL, -- e.g., '2023/2024'
            logo VARCHAR(255),
            manager_id INT,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            FOREIGN KEY (manager_id) REFERENCES users(id) ON DELETE SET NULL
        ) ENGINE=InnoDB;

        -- Clubs Table
        CREATE TABLE IF NOT EXISTS clubs (
            id INT AUTO_INCREMENT PRIMARY KEY,
            name VARCHAR(255) NOT NULL UNIQUE,
            logo VARCHAR(255),
            stadium VARCHAR(255),
            league_id INT,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            FOREIGN KEY (league_id) REFERENCES leagues(id) ON DELETE CASCADE
        ) ENGINE=InnoDB;

        -- Club Managers Table (Junction Table)
        -- A club can have multiple managers (CEO, President), and a user could manage multiple clubs.
        CREATE TABLE IF NOT EXISTS club_managers (
            user_id INT NOT NULL,
            club_id INT NOT NULL,
            role_title VARCHAR(100) NOT NULL COMMENT 'e.g., President, CEO, Finance Manager',
            PRIMARY KEY (user_id, club_id),
            FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
            FOREIGN KEY (club_id) REFERENCES clubs(id) ON DELETE CASCADE
        ) ENGINE=InnoDB;

        -- Teams Table: Represents specific teams within a club (e.g., Men's Senior, Women's U21)
        CREATE TABLE IF NOT EXISTS teams (
            id INT AUTO_INCREMENT PRIMARY KEY,
            club_id INT NOT NULL,
            name VARCHAR(255) NOT NULL,
            manager_id INT, -- The Head Coach / Team Manager
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            FOREIGN KEY (club_id) REFERENCES clubs(id) ON DELETE CASCADE,
            FOREIGN KEY (manager_id) REFERENCES users(id) ON DELETE SET NULL,
            UNIQUE(club_id, name)
        ) ENGINE=InnoDB;

        -- Players Table: Links a user to a team and adds player-specific data
        CREATE TABLE IF NOT EXISTS players (
            user_id INT PRIMARY KEY, -- A player IS a user
            team_id INT,
            position VARCHAR(50),
            price DECIMAL(4,1) DEFAULT 4.0,
            squad_number INT,
            height_cm INT,
            weight_kg INT,
            nationality VARCHAR(100),
            date_of_birth DATE,
            FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
            FOREIGN KEY (team_id) REFERENCES teams(id) ON DELETE SET NULL
        ) ENGINE=InnoDB;

        -- Player Injuries Table
        CREATE TABLE IF NOT EXISTS player_injuries (
            id INT AUTO_INCREMENT PRIMARY KEY,
            player_id INT NOT NULL,
            injury_description VARCHAR(255) NOT NULL,
            start_date DATE NOT NULL,
            expected_return_date DATE NULL,
            status ENUM('Active', 'Recovering', 'Fit') DEFAULT 'Active',
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            FOREIGN KEY (player_id) REFERENCES users(id) ON DELETE CASCADE
        ) ENGINE=InnoDB;

        -- Player Suspensions Table
        CREATE TABLE IF NOT EXISTS player_suspensions (
            id INT AUTO_INCREMENT PRIMARY KEY,
            player_id INT NOT NULL,
            fixture_incurred_id INT NULL,
            reason VARCHAR(255) NOT NULL,
            matches_to_serve INT NOT NULL,
            status ENUM('Active', 'Served') DEFAULT 'Active',
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            FOREIGN KEY (player_id) REFERENCES users(id) ON DELETE CASCADE,
            FOREIGN KEY (fixture_incurred_id) REFERENCES fixtures(id) ON DELETE SET NULL
        ) ENGINE=InnoDB;

        -- Training Sessions Table
        CREATE TABLE IF NOT EXISTS training_sessions (
            id INT AUTO_INCREMENT PRIMARY KEY,
            team_id INT NOT NULL,
            session_date DATE NOT NULL,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            FOREIGN KEY (team_id) REFERENCES teams(id) ON DELETE CASCADE
        ) ENGINE=InnoDB;

        -- Fixtures Table
        CREATE TABLE IF NOT EXISTS fixtures (
            id INT AUTO_INCREMENT PRIMARY KEY,
            league_id INT NOT NULL,
            home_team_id INT NOT NULL,
            away_team_id INT NOT NULL,
            match_datetime DATETIME NOT NULL,
            gameweek INT,
            venue VARCHAR(255),
            status ENUM('scheduled', 'live', 'postponed', 'cancelled', 'finished') DEFAULT 'scheduled',
            home_team_score INT DEFAULT 0,
            away_team_score INT DEFAULT 0,
            commissioner_id INT,
            referee_id INT,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            FOREIGN KEY (league_id) REFERENCES leagues(id) ON DELETE CASCADE,
            FOREIGN KEY (home_team_id) REFERENCES teams(id) ON DELETE CASCADE,
            FOREIGN KEY (away_team_id) REFERENCES teams(id) ON DELETE CASCADE,
            FOREIGN KEY (commissioner_id) REFERENCES users(id) ON DELETE SET NULL,
            FOREIGN KEY (referee_id) REFERENCES users(id) ON DELETE SET NULL
        ) ENGINE=InnoDB;

        -- Match Lineups Table: Stores the players selected for a specific match
        CREATE TABLE IF NOT EXISTS match_lineups (
            id INT AUTO_INCREMENT PRIMARY KEY,
            fixture_id INT NOT NULL,
            team_id INT NOT NULL,
            player_id INT NOT NULL,
            squad_number INT,
            formation VARCHAR(20) NULL, -- e.g., '4-4-2', '4-3-3'
            status ENUM('starter', 'substitute') NOT NULL,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            FOREIGN KEY (fixture_id) REFERENCES fixtures(id) ON DELETE CASCADE,
            FOREIGN KEY (team_id) REFERENCES teams(id) ON DELETE CASCADE,
            FOREIGN KEY (player_id) REFERENCES users(id) ON DELETE CASCADE,
            UNIQUE(fixture_id, player_id)
        ) ENGINE=InnoDB;

        -- Match Events Table: Logs goals, cards, substitutions, etc.
        CREATE TABLE IF NOT EXISTS match_events (
            id INT AUTO_INCREMENT PRIMARY KEY,
            fixture_id INT NOT NULL,
            team_id INT NOT NULL,
            player_id INT NULL, -- Can be null for team events like corners
            event_type ENUM(
                'goal', 'assist', 'yellow_card', 'red_card', 'substitution_in', 'substitution_out',
                'corner', 'free_kick', 'shot_on_target', 'shot_off_target', 'foul',
                'penalty_scored', 'penalty_missed', 'own_goal'
            ) NOT NULL,
            minute INT NOT NULL,
            assisted_by_player_id INT NULL,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            FOREIGN KEY (fixture_id) REFERENCES fixtures(id) ON DELETE CASCADE,
            FOREIGN KEY (team_id) REFERENCES teams(id) ON DELETE CASCADE,
            FOREIGN KEY (player_id) REFERENCES users(id) ON DELETE CASCADE,
            FOREIGN KEY (assisted_by_player_id) REFERENCES users(id) ON DELETE SET NULL
        ) ENGINE=InnoDB;

        -- Match Reports Table: For referees to submit post-match notes
        CREATE TABLE IF NOT EXISTS match_reports (
            id INT AUTO_INCREMENT PRIMARY KEY,
            fixture_id INT NOT NULL,
            referee_id INT NOT NULL,
            report_content TEXT NOT NULL,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            FOREIGN KEY (fixture_id) REFERENCES fixtures(id) ON DELETE CASCADE,
            FOREIGN KEY (referee_id) REFERENCES users(id) ON DELETE CASCADE,
            UNIQUE(fixture_id, referee_id)
        ) ENGINE=InnoDB;

        -- Fantasy Teams Table: Each user has one fantasy team per season
        CREATE TABLE IF NOT EXISTS fantasy_teams (
            id INT AUTO_INCREMENT PRIMARY KEY,
            user_id INT NOT NULL,
            name VARCHAR(255) NOT NULL,
            season VARCHAR(50) NOT NULL,
            total_points INT DEFAULT 0,
            bank DECIMAL(4,1) DEFAULT 100.0,
            gameweek_points INT DEFAULT 0,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
            UNIQUE(user_id, season)
        ) ENGINE=InnoDB;

        -- Fantasy Squads Table: Links a fantasy team to the real players
        CREATE TABLE IF NOT EXISTS fantasy_squads (
            fantasy_team_id INT NOT NULL,
            player_id INT NOT NULL,
            is_captain BOOLEAN DEFAULT FALSE,
            is_vice_captain BOOLEAN DEFAULT FALSE,
            is_starter BOOLEAN DEFAULT FALSE,
            bench_order TINYINT NULL,
            gameweek_added INT NOT NULL,
            PRIMARY KEY (fantasy_team_id, player_id),
            FOREIGN KEY (fantasy_team_id) REFERENCES fantasy_teams(id) ON DELETE CASCADE,
            FOREIGN KEY (player_id) REFERENCES users(id) ON DELETE CASCADE
        ) ENGINE=InnoDB;

        -- Fantasy Leagues Table: User-created private or public leagues
        CREATE TABLE IF NOT EXISTS fantasy_leagues (
            id INT AUTO_INCREMENT PRIMARY KEY,
            name VARCHAR(255) NOT NULL,
            creator_user_id INT NOT NULL,
            season VARCHAR(50) NOT NULL,
            join_code VARCHAR(10) NOT NULL UNIQUE,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            FOREIGN KEY (creator_user_id) REFERENCES users(id) ON DELETE CASCADE
        ) ENGINE=InnoDB;

        -- Fantasy League Members Table: Junction table for teams and leagues
        CREATE TABLE IF NOT EXISTS fantasy_league_members (
            league_id INT NOT NULL,
            fantasy_team_id INT NOT NULL,
            PRIMARY KEY (league_id, fantasy_team_id),
            FOREIGN KEY (league_id) REFERENCES fantasy_leagues(id) ON DELETE CASCADE,
            FOREIGN KEY (fantasy_team_id) REFERENCES fantasy_teams(id) ON DELETE CASCADE
        ) ENGINE=InnoDB;

        -- Fantasy Transfers Log
        CREATE TABLE IF NOT EXISTS fantasy_transfers (
            id INT AUTO_INCREMENT PRIMARY KEY,
            fantasy_team_id INT NOT NULL,
            gameweek INT NOT NULL,
            player_out_id INT NOT NULL,
            player_in_id INT NOT NULL,
            transfer_time TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            FOREIGN KEY (fantasy_team_id) REFERENCES fantasy_teams(id) ON DELETE CASCADE,
            FOREIGN KEY (player_out_id) REFERENCES users(id) ON DELETE CASCADE,
            FOREIGN KEY (player_in_id) REFERENCES users(id) ON DELETE CASCADE
        ) ENGINE=InnoDB;

        -- Fantasy Player Points (per gameweek)
        CREATE TABLE IF NOT EXISTS fantasy_player_points (
            player_id INT NOT NULL,
            gameweek INT NOT NULL,
            fixture_id INT NOT NULL,
            points INT NOT NULL,
            minutes_played INT DEFAULT 0,
            breakdown JSON,
            PRIMARY KEY (player_id, gameweek, fixture_id),
            FOREIGN KEY (player_id) REFERENCES users(id) ON DELETE CASCADE,
            FOREIGN KEY (fixture_id) REFERENCES fixtures(id) ON DELETE CASCADE
        ) ENGINE=InnoDB;

        -- Gameweeks Table: Stores deadlines for each gameweek
        CREATE TABLE IF NOT EXISTS gameweeks (
            id INT AUTO_INCREMENT PRIMARY KEY,
            league_id INT NOT NULL,
            season_name VARCHAR(50) NOT NULL,
            gameweek_number INT NOT NULL,
            deadline_time DATETIME,
            is_finished BOOLEAN DEFAULT FALSE,
            FOREIGN KEY (league_id) REFERENCES leagues(id) ON DELETE CASCADE,
            UNIQUE(league_id, season_name, gameweek_number)
        ) ENGINE=InnoDB;

        -- Settings Table
        CREATE TABLE IF NOT EXISTS settings (
            setting_key VARCHAR(50) PRIMARY KEY,
            setting_value TEXT,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
        ) ENGINE=InnoDB;
    ";

    // Execute the multi-query
    $pdo->exec($sql);

    echo "All base tables created successfully or already exist.<br>";

    // --- Schema Migrations ---
    // This section ensures existing tables are up-to-date.

    // Check for 'formation' column in 'match_lineups' and add if it doesn't exist.
    $stmt = $pdo->query("SHOW COLUMNS FROM `match_lineups` LIKE 'formation'");
    if ($stmt->rowCount() == 0) {
        $pdo->exec("ALTER TABLE `match_lineups` ADD `formation` VARCHAR(20) NULL AFTER `squad_number`");
        echo "-> Added 'formation' column to 'match_lineups' table.<br>";
    } else {
        echo "-> 'formation' column already exists in 'match_lineups'.<br>";
    }

    // Check for 'gameweek' column in 'fixtures' and add if it doesn't exist.
    $stmt = $pdo->query("SHOW COLUMNS FROM `fixtures` LIKE 'gameweek'");
    if ($stmt->rowCount() == 0) {
        $pdo->exec("ALTER TABLE `fixtures` ADD `gameweek` INT NULL AFTER `match_datetime`");
        echo "-> Added 'gameweek' column to 'fixtures' table.<br>";
    } else {
        echo "-> 'gameweek' column already exists in 'fixtures'.<br>";
    }

    // Check for 'season' column in 'leagues' and add if it doesn't exist.
    $stmt = $pdo->query("SHOW COLUMNS FROM `leagues` LIKE 'season'");
    if ($stmt->rowCount() == 0) {
        $pdo->exec("ALTER TABLE `leagues` ADD `season` VARCHAR(50) NOT NULL AFTER `country`");
        echo "-> Added 'season' column to 'leagues' table.<br>";
    } else {
        echo "-> 'season' column already exists in 'leagues'.<br>";
    }

    // Check for 'price' column in 'players' and add if it doesn't exist.
    $stmt = $pdo->query("SHOW COLUMNS FROM `players` LIKE 'price'");
    if ($stmt->rowCount() == 0) {
        $pdo->exec("ALTER TABLE `players` ADD `price` DECIMAL(4,1) DEFAULT 4.0 AFTER `position`");
        echo "-> Added 'price' column to 'players' table.<br>";
    } else {
        echo "-> 'price' column already exists in 'players'.<br>";
    }

    // Update event_type ENUM in match_events
    $pdo->exec("ALTER TABLE `match_events` MODIFY `event_type` ENUM(
        'goal', 'assist', 'yellow_card', 'red_card', 'substitution_in', 'substitution_out',
        'corner', 'free_kick', 'shot_on_target', 'shot_off_target', 'foul',
        'penalty_scored', 'penalty_missed', 'own_goal'
    ) NOT NULL");
    echo "-> Updated 'event_type' ENUM in 'match_events' table.<br>";

    // Add timer columns to fixtures table
    $stmt = $pdo->query("SHOW COLUMNS FROM `fixtures` LIKE 'first_half_start_time'");
    if ($stmt->rowCount() == 0) {
        $pdo->exec("ALTER TABLE `fixtures` 
            ADD `first_half_start_time` DATETIME NULL DEFAULT NULL AFTER `status`,
            ADD `second_half_start_time` DATETIME NULL DEFAULT NULL AFTER `first_half_start_time`");
        echo "-> Added timer columns to 'fixtures' table.<br>";
    }

    // Add is_starter to fantasy_squads
    $stmt = $pdo->query("SHOW COLUMNS FROM `fantasy_squads` LIKE 'is_starter'");
    if ($stmt->rowCount() == 0) {
        $pdo->exec("ALTER TABLE `fantasy_squads` ADD `is_starter` BOOLEAN DEFAULT FALSE AFTER `is_vice_captain`");
        echo "-> Added 'is_starter' column to 'fantasy_squads' table.<br>";
    } else {
        echo "-> 'is_starter' column already exists in 'fantasy_squads'.<br>";
    }

    // Add bench_order to fantasy_squads
    $stmt = $pdo->query("SHOW COLUMNS FROM `fantasy_squads` LIKE 'bench_order'");
    if ($stmt->rowCount() == 0) {
        $pdo->exec("ALTER TABLE `fantasy_squads` ADD `bench_order` TINYINT NULL AFTER `is_starter`");
        echo "-> Added 'bench_order' column to 'fantasy_squads' table.<br>";
    } else {
        echo "-> 'bench_order' column already exists in 'fantasy_squads'.<br>";
    }

    // Add fantasy_player_points table
    $stmt = $pdo->query("SHOW TABLES LIKE 'fantasy_player_points'");
    if ($stmt->rowCount() == 0) {
        $pdo->exec("CREATE TABLE fantasy_player_points (player_id INT NOT NULL, gameweek INT NOT NULL, fixture_id INT NOT NULL, points INT NOT NULL, minutes_played INT DEFAULT 0, breakdown JSON, PRIMARY KEY (player_id, gameweek, fixture_id), FOREIGN KEY (player_id) REFERENCES users(id) ON DELETE CASCADE, FOREIGN KEY (fixture_id) REFERENCES fixtures(id) ON DELETE CASCADE) ENGINE=InnoDB;");
        echo "-> Created 'fantasy_player_points' table.<br>";
    } else {
        echo "-> 'fantasy_player_points' table already exists.<br>";
    }

    // Add bank to fantasy_teams
    $stmt = $pdo->query("SHOW COLUMNS FROM `fantasy_teams` LIKE 'bank'");
    if ($stmt->rowCount() == 0) {
        $pdo->exec("ALTER TABLE `fantasy_teams` ADD `bank` DECIMAL(4,1) DEFAULT 100.0 AFTER `total_points`");
        echo "-> Added 'bank' column to 'fantasy_teams' table.<br>";
    }

    // Add fantasy_transfers table
    $stmt = $pdo->query("SHOW TABLES LIKE 'fantasy_transfers'");
    if ($stmt->rowCount() == 0) {
        $pdo->exec("CREATE TABLE fantasy_transfers (id INT AUTO_INCREMENT PRIMARY KEY, fantasy_team_id INT NOT NULL, gameweek INT NOT NULL, player_out_id INT NOT NULL, player_in_id INT NOT NULL, transfer_time TIMESTAMP DEFAULT CURRENT_TIMESTAMP, FOREIGN KEY (fantasy_team_id) REFERENCES fantasy_teams(id) ON DELETE CASCADE, FOREIGN KEY (player_out_id) REFERENCES users(id) ON DELETE CASCADE, FOREIGN KEY (player_in_id) REFERENCES users(id) ON DELETE CASCADE) ENGINE=InnoDB;");
        echo "-> Created 'fantasy_transfers' table.<br>";
    } else {
        echo "-> 'fantasy_transfers' table already exists.<br>";
    }

    // Add settings table
    $stmt = $pdo->query("SHOW TABLES LIKE 'settings'");
    if ($stmt->rowCount() == 0) {
        $pdo->exec("CREATE TABLE settings (setting_key VARCHAR(50) PRIMARY KEY, setting_value TEXT, created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP, updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP) ENGINE=InnoDB;");
        echo "-> Created 'settings' table.<br>";
    }

    // Add gameweeks table
    $stmt = $pdo->query("SHOW TABLES LIKE 'gameweeks'");
    if ($stmt->rowCount() == 0) {
        $pdo->exec("CREATE TABLE gameweeks (id INT AUTO_INCREMENT PRIMARY KEY, league_id INT NOT NULL, season_name VARCHAR(50) NOT NULL, gameweek_number INT NOT NULL, deadline_time DATETIME, is_finished BOOLEAN DEFAULT FALSE, FOREIGN KEY (league_id) REFERENCES leagues(id) ON DELETE CASCADE, UNIQUE(league_id, season_name, gameweek_number)) ENGINE=InnoDB;");
        echo "-> Created 'gameweeks' table.<br>";
    } else {
        echo "-> 'gameweeks' table already exists.<br>";
    }


    echo "<h4>Database schema is up-to-date.</h4>";

} catch (PDOException $e) {
    die("DB ERROR: " . $e->getMessage());
}

?>