<?php
require_once '../core/bootstrap.php';

// --- Authentication Check ---
if (!isset($_SESSION['user_id']) || $_SESSION['user_role'] !== 'referee') {
    header('Location: login.php');
    exit();
}

$pageTitle = 'My Matches';
include 'includes/header.php';

$referee_id = $_SESSION['user_id'];
$db = new Database();

// Fetch assigned matches
$db->query("
    SELECT f.*, 
           t1.name as home_team_name, c1.logo as home_team_logo,
           t2.name as away_team_name, c2.logo as away_team_logo,
           l.name as league_name
    FROM fixtures f
    JOIN teams t1 ON f.home_team_id = t1.id
    JOIN clubs c1 ON t1.club_id = c1.id
    JOIN teams t2 ON f.away_team_id = t2.id
    JOIN clubs c2 ON t2.club_id = c2.id
    JOIN leagues l ON f.league_id = l.id
    WHERE f.referee_id = :referee_id
    ORDER BY f.match_datetime DESC
");
$db->bind(':referee_id', $referee_id);
$matches = $db->resultSet();

$upcoming = [];
$past = [];

foreach ($matches as $match) {
    if (in_array($match->status, ['finished', 'abandoned'])) {
        $past[] = $match;
    } else {
        $upcoming[] = $match;
    }
}

// Sort upcoming ascending (soonest first)
usort($upcoming, fn($a, $b) => strtotime($a->match_datetime) - strtotime($b->match_datetime));
?>

<div class="page-content">
    <div class="matches-hero" style="background: linear-gradient(135deg, #2c3e50 0%, #4ca1af 100%);">
        <h1>Referee Dashboard</h1>
        <p style="color: rgba(255,255,255,0.8); margin: 0;">Manage your assigned fixtures and submit reports.</p>
    </div>

    <div class="dashboard-section">
        <h2 class="dashboard-title">Upcoming Assignments</h2>
        <?php if (empty($upcoming)): ?>
            <p class="no-data-msg">No upcoming matches assigned.</p>
        <?php else: ?>
            <div class="fixture-list">
                <?php foreach ($upcoming as $match): ?>
                    <div class="fixture-item">
                        <div class="team home-team">
                            <span class="team-name"><?php echo htmlspecialchars($match->home_team_name); ?></span>
                            <img src="<?php echo htmlspecialchars($match->home_team_logo); ?>" alt="" class="team-logo">
                        </div>
                        <div class="score-box time-box">
                            <span class="time"><?php echo date('H:i', strtotime($match->match_datetime)); ?></span>
                            <span class="date"><?php echo date('d M', strtotime($match->match_datetime)); ?></span>
                        </div>
                        <div class="team away-team">
                            <img src="<?php echo htmlspecialchars($match->away_team_logo); ?>" alt="" class="team-logo">
                            <span class="team-name"><?php echo htmlspecialchars($match->away_team_name); ?></span>
                        </div>
                    </div>
                    <div style="text-align: center; margin-bottom: 15px;">
                        <span class="badge" style="background: <?php echo ($match->referee_payment_status ?? 'pending') === 'paid' ? '#2ecc71' : '#f1c40f'; ?>; color: white; padding: 4px 8px; border-radius: 4px; font-size: 0.8rem; margin-right: 10px; vertical-align: middle;">
                            Fee: <?php echo ucfirst($match->referee_payment_status ?? 'Pending'); ?>
                        </span>
                        <a href="fixture.php?id=<?php echo $match->id; ?>" class="btn-sm">View Details</a>
                        <?php if ($match->status === 'live' || $match->status === 'half_time'): ?>
                            <a href="submit_report.php?id=<?php echo $match->id; ?>" class="btn-sm" style="background-color: #e74c3c; color: white;">Report Incident</a>
                        <?php endif; ?>
                    </div>
                <?php endforeach; ?>
            </div>
        <?php endif; ?>
    </div>

    <div class="dashboard-section">
        <h2 class="dashboard-title">Match History & Reports</h2>
        <?php if (empty($past)): ?>
            <p class="no-data-msg">No past matches found.</p>
        <?php else: ?>
            <div class="table-wrapper">
                <table class="styled-table">
                    <thead>
                        <tr>
                            <th>Date</th>
                            <th>Match</th>
                            <th>Result</th>
                            <th>Fee Status</th>
                            <th>Action</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($past as $match): ?>
                            <tr>
                                <td><?php echo date('d M Y', strtotime($match->match_datetime)); ?></td>
                                <td><?php echo htmlspecialchars($match->home_team_name . ' vs ' . $match->away_team_name); ?></td>
                                <td><?php echo $match->home_team_score . ' - ' . $match->away_team_score; ?></td>
                                <td>
                                    <span class="badge" style="background: <?php echo ($match->referee_payment_status ?? 'pending') === 'paid' ? '#2ecc71' : '#f1c40f'; ?>; color: white; padding: 4px 8px; border-radius: 4px; font-size: 0.8rem;">
                                        <?php echo ucfirst($match->referee_payment_status ?? 'Pending'); ?>
                                    </span>
                                </td>
                                <td>
                                    <a href="submit_report.php?id=<?php echo $match->id; ?>" class="btn-sm">Submit Report</a>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        <?php endif; ?>
    </div>
</div>

<style>
/* Fixture List Styles */
.fixture-list {
    display: flex;
    flex-direction: column;
    gap: 10px;
}
.fixture-item {
    display: flex;
    justify-content: space-between;
    align-items: center;
    padding: 15px;
    background: white;
    border-radius: 8px;
    text-decoration: none;
    color: inherit;
    border: 1px solid #eee;
}
.fixture-item .team {
    display: flex;
    align-items: center;
    gap: 10px;
    flex: 1;
}
.fixture-item .home-team {
    justify-content: flex-end;
    text-align: right;
}
.fixture-item .away-team {
    justify-content: flex-start;
    text-align: left;
}
.fixture-item .team-logo {
    width: 40px;
    height: 40px;
    object-fit: contain;
}
</style>

<?php include 'includes/footer.php'; ?>