<?php

class MatchEvent
{
    private $db;

    public function __construct()
    {
        $this->db = new Database;
    }

    public function log($data)
    {
        $this->db->query('INSERT INTO match_events (fixture_id, team_id, player_id, event_type, minute, assisted_by_player_id) VALUES (:fixture_id, :team_id, :player_id, :event_type, :minute, :assisted_by_player_id)');
        $this->db->bind(':fixture_id', $data['fixture_id']);
        $this->db->bind(':team_id', $data['team_id']);

        $this->db->bind(':player_id', $data['player_id'] ?? null, PDO::PARAM_INT);
        $this->db->bind(':event_type', $data['event_type']);
        $this->db->bind(':minute', $data['minute']);
        $this->db->bind(':assisted_by_player_id', $data['assisted_by_player_id'] ?? null, PDO::PARAM_INT);

        if ($this->db->execute()) {
            return $this->db->lastInsertId();
        }
        return false;
    }

    /**
     * Fetches all events for a given fixture, ordered for timeline display.
     *
     * @param int $fixture_id The ID of the fixture.
     * @return array An array of event objects.
     */
    public function getEventsForFixture(int $fixture_id): array
    {
        $this->db->query("SELECT * FROM match_events WHERE fixture_id = :fixture_id ORDER BY minute ASC, id ASC");
        $this->db->bind(':fixture_id', $fixture_id);
        return $this->db->resultSet();
    }

    public function getEventById(int $event_id)
    {
        $this->db->query("SELECT * FROM match_events WHERE id = :id");
        $this->db->bind(':id', $event_id);
        return $this->db->single();
    }

    public function getLatestEventForFixture(int $fixture_id)
    {
        $this->db->query("SELECT * FROM match_events WHERE fixture_id = :fixture_id ORDER BY id DESC LIMIT 1");
        $this->db->bind(':fixture_id', $fixture_id);
        return $this->db->single();
    }

    public function deleteById(int $event_id)
    {
        $this->db->query("DELETE FROM match_events WHERE id = :id");
        $this->db->bind(':id', $event_id);
        return $this->db->execute();
    }

    public function getStatsForFixture($fixture_id)
    {
        $this->db->query("
            SELECT
                me.team_id,
                SUM(CASE WHEN me.event_type IN ('goal', 'penalty_scored', 'own_goal') THEN 1 ELSE 0 END) AS goals,
                SUM(CASE WHEN me.event_type = 'shot_on_target' THEN 1 ELSE 0 END) AS shots_on_target,
                SUM(CASE WHEN me.event_type = 'shot_off_target' THEN 1 ELSE 0 END) AS shots_off_target,
                SUM(CASE WHEN me.event_type = 'corner' THEN 1 ELSE 0 END) AS corners,
                SUM(CASE WHEN me.event_type = 'foul' THEN 1 ELSE 0 END) AS fouls,
                SUM(CASE WHEN me.event_type = 'yellow_card' THEN 1 ELSE 0 END) AS yellow_cards,
                SUM(CASE WHEN me.event_type = 'red_card' THEN 1 ELSE 0 END) AS red_cards
            FROM match_events me
            WHERE me.fixture_id = :fixture_id
            GROUP BY me.team_id
        ");
        $this->db->bind(':fixture_id', $fixture_id);
        return $this->db->resultSet();
    }
}
