<?php
$wallet_id_filter = filter_input(INPUT_GET, 'wallet_id', FILTER_VALIDATE_INT);
$pageTitle = 'Wallet Transactions Log';
if ($wallet_id_filter) {
    $pageTitle = "Transactions for Wallet #$wallet_id_filter";
}

include 'includes/header.php';

// --- Admin Check ---
$current_role = $_SESSION['user_role'] ?? $_SESSION['admin_user_role'] ?? '';
if (!in_array($current_role, ['admin', 'super_admin', 'finance_manager'])) {
    header('Location: dashboard.php');
    exit();
}

$walletModel = new Wallet();
$settingModel = new Setting();

// Pagination
$page = filter_input(INPUT_GET, 'page', FILTER_VALIDATE_INT) ?: 1;
$limit = 50;
$offset = ($page - 1) * $limit;

// Fetch Data
if ($wallet_id_filter) {
    $total_transactions = $walletModel->getTransactionsCountByWalletId($wallet_id_filter);
    $transactions = $walletModel->getTransactionsByWalletIdWithDetails($wallet_id_filter, $limit, $offset);
} else {
    $total_transactions = $walletModel->getTotalSystemTransactionsCount();
    $transactions = $walletModel->getAllSystemTransactions($limit, $offset);
}
$total_pages = ceil($total_transactions / $limit);

// Get currency symbol
$settings = $settingModel->getAll();
$currency_symbol = ($settings['system_currency'] ?? 'USD') === 'ZMW' ? 'K' : '$';

include 'includes/sidebar.php';
?>

<div class="page-content">
    <div class="matches-header">
        <h1><?php echo $pageTitle; ?></h1>
        <div class="header-actions">
            <?php if ($wallet_id_filter): ?>
                <a href="wallet_transactions.php" class="btn btn-secondary">View All Logs</a>
            <?php endif; ?>
            <a href="wallet_management.php" class="btn btn-secondary">Wallet Management</a>
        </div>
    </div>

    <div class="dashboard-section">
        <div class="table-wrapper">
            <table class="styled-table">
                <thead>
                    <tr>
                        <th>ID</th>
                        <th>Date</th>
                        <th>Entity</th>
                        <th>Type</th>
                        <th>Amount</th>
                        <th>Description</th>
                        <th>Status</th>
                    </tr>
                </thead>
                <tbody>
                    <?php if (empty($transactions)): ?>
                        <tr><td colspan="7" style="text-align:center;">No transactions found.</td></tr>
                    <?php else: ?>
                        <?php foreach ($transactions as $t): ?>
                            <tr>
                                <td>#<?php echo $t->id; ?></td>
                                <td><?php echo date('d M Y, H:i', strtotime($t->created_at)); ?></td>
                                <td>
                                    <?php 
                                    if ($t->entity_type === 'user') {
                                        echo htmlspecialchars($t->first_name . ' ' . $t->last_name);
                                        echo ' <small class="text-muted">(' . htmlspecialchars($t->user_email) . ')</small>';
                                    } elseif ($t->entity_type === 'club') {
                                        echo htmlspecialchars($t->club_name);
                                        echo ' <small class="text-muted">(Club)</small>';
                                    } else {
                                        echo ucfirst($t->entity_type) . ' #' . $t->entity_id;
                                    }
                                    ?>
                                </td>
                                <td><span class="badge badge-<?php echo $t->type; ?>"><?php echo ucfirst(str_replace('_', ' ', $t->type)); ?></span></td>
                                <td class="<?php echo in_array($t->type, ['deposit', 'transfer_in']) ? 'text-success' : 'text-danger'; ?>">
                                    <?php echo (in_array($t->type, ['deposit', 'transfer_in']) ? '+' : '-') . $currency_symbol . number_format($t->amount, 2); ?>
                                </td>
                                <td><?php echo htmlspecialchars($t->description); ?></td>
                                <td><?php echo ucfirst($t->status); ?></td>
                            </tr>
                        <?php endforeach; ?>
                    <?php endif; ?>
                </tbody>
            </table>
        </div>

        <!-- Pagination -->
        <?php if ($total_pages > 1): ?>
            <div class="pagination-controls">
                <?php 
                    $query_string = $wallet_id_filter ? "&wallet_id=" . $wallet_id_filter : "";
                ?>
                <?php if ($page > 1): ?>
                    <a href="?page=<?php echo $page - 1; ?><?php echo $query_string; ?>" class="btn-sm">Previous</a>
                <?php endif; ?>
                <span class="page-info">Page <?php echo $page; ?> of <?php echo $total_pages; ?></span>
                <?php if ($page < $total_pages): ?>
                    <a href="?page=<?php echo $page + 1; ?><?php echo $query_string; ?>" class="btn-sm">Next</a>
                <?php endif; ?>
            </div>
        <?php endif; ?>
    </div>
</div>

<style>
.text-muted { color: #6c757d; font-size: 0.85rem; }
.pagination-controls { display: flex; justify-content: center; gap: 10px; margin-top: 20px; align-items: center; }
</style>

<?php include 'includes/footer.php'; ?>