<?php
$pageTitle = 'Fine Payments History';
include 'includes/header.php';

// --- Admin Check ---
if (!in_array($_SESSION['admin_user_role'] ?? '', ['admin', 'super_admin'])) {
    header('Location: dashboard.php');
    exit();
}

$db = new Database();
$settingModel = new Setting();

// Pagination
$page = filter_input(INPUT_GET, 'page', FILTER_VALIDATE_INT) ?: 1;
$limit = 20;
$offset = ($page - 1) * $limit;

// Count total paid fines
$db->query("SELECT COUNT(*) as count FROM fines WHERE status = 'paid'");
$total_fines = $db->single()->count;
$total_pages = ceil($total_fines / $limit);

// Fetch paid fines
$db->query("
    SELECT f.*, 
           CASE WHEN f.entity_type = 'player' THEN CONCAT(u.first_name, ' ', u.last_name) 
                WHEN f.entity_type = 'club' THEN c.name 
           END as entity_name,
           u.email as player_email
    FROM fines f
    LEFT JOIN users u ON f.entity_id = u.id AND f.entity_type = 'player'
    LEFT JOIN clubs c ON f.entity_id = c.id AND f.entity_type = 'club'
    WHERE f.status = 'paid'
    ORDER BY f.paid_date DESC
    LIMIT :limit OFFSET :offset
");
$db->bind(':limit', $limit);
$db->bind(':offset', $offset);
$paid_fines = $db->resultSet();

// Get currency symbol
$settings = $settingModel->getAll();
$currency_symbol = ($settings['system_currency'] ?? 'USD') === 'ZMW' ? 'K' : '$';

include 'includes/sidebar.php';
?>

<div class="page-content">
    <div class="matches-header">
        <h1>Fine Payments History</h1>
        <a href="manage_fines.php" class="btn btn-secondary"><i class="fa-solid fa-arrow-left"></i> Back to Manage Fines</a>
    </div>

    <div class="dashboard-section">
        <div class="table-wrapper">
            <table class="styled-table">
                <thead>
                    <tr>
                        <th>Date Paid</th>
                        <th>Entity</th>
                        <th>Type</th>
                        <th>Reason</th>
                        <th>Amount</th>
                        <th>Issued Date</th>
                    </tr>
                </thead>
                <tbody>
                    <?php if (empty($paid_fines)): ?>
                        <tr><td colspan="6" style="text-align:center;">No paid fines found.</td></tr>
                    <?php else: ?>
                        <?php foreach ($paid_fines as $fine): ?>
                            <tr>
                                <td><?php echo date('d M Y, H:i', strtotime($fine->paid_date)); ?></td>
                                <td>
                                    <?php echo htmlspecialchars($fine->entity_name); ?>
                                    <?php if ($fine->entity_type === 'player'): ?>
                                        <br><small class="text-muted"><?php echo htmlspecialchars($fine->player_email); ?></small>
                                    <?php endif; ?>
                                </td>
                                <td><span class="badge badge-info"><?php echo ucfirst($fine->entity_type); ?></span></td>
                                <td><?php echo htmlspecialchars($fine->reason); ?></td>
                                <td><strong><?php echo $currency_symbol . number_format($fine->amount, 2); ?></strong></td>
                                <td><?php echo date('d M Y', strtotime($fine->issued_date)); ?></td>
                            </tr>
                        <?php endforeach; ?>
                    <?php endif; ?>
                </tbody>
            </table>
        </div>

        <!-- Pagination -->
        <?php if ($total_pages > 1): ?>
            <div class="pagination-controls">
                <?php if ($page > 1): ?>
                    <a href="?page=<?php echo $page - 1; ?>" class="btn-sm">Previous</a>
                <?php endif; ?>
                <span class="page-info">Page <?php echo $page; ?> of <?php echo $total_pages; ?></span>
                <?php if ($page < $total_pages): ?>
                    <a href="?page=<?php echo $page + 1; ?>" class="btn-sm">Next</a>
                <?php endif; ?>
            </div>
        <?php endif; ?>
    </div>
</div>

<style>
.badge-info { background-color: #17a2b8; color: white; padding: 4px 8px; border-radius: 4px; font-size: 0.8rem; }
.text-muted { color: #6c757d; font-size: 0.85rem; }
.pagination-controls { display: flex; justify-content: center; gap: 10px; margin-top: 20px; align-items: center; }
</style>

<?php include 'includes/footer.php'; ?>